import { encode as base64Encode, decode as base64Decode } from 'base-64';
import axios from 'axios';

/**
 * Resource Parameter Structure
 */
export interface ResourceParam {
    path?: string;
    [key: string]: any;
}

/**
 * HTTP Method Enum
 */
export enum HttpMethod {
    GET = 'GET',
    POST = 'POST',
    PUT = 'PUT',
    DELETE = 'DELETE',
}

/**
 * Resource Structure
 */
export interface Resource {
    type?: string;
    name?: string;
    param?: ResourceParam;
    [key: string]: any;
}

/**
 * Subject Structure
 */
export interface Subject {
    [key: string]: any;
}

/**
 * Environment Structure
 */
export interface Environment {
    [key: string]: any;
}

/**
 * Policy Request Configuration
 */
export interface PolicyConfig {
    resource?: Resource;
    action?: string;
}

/**
 * Policy Request Data Structure
 */
export interface PolicyRequestData {
    subject: Subject;
    resource: Resource;
    action: string;
    environment: Environment;
}

/**
 * Get public IP address
 * @returns Public IP address or null if failed
 */
async function getPublicIP(): Promise<string | null> {
    try {
        const response = await axios.get('https://api.ipify.org/?format=json', {
            timeout: 5000, // 5 second timeout
        });
        const ip = response.data?.ip;
        if (!ip || typeof ip !== 'string') {
            throw new Error('IP address not found in response');
        }
        return ip;
    } catch (error) {
        console.error('Failed to get public IP:', error);
        return null;
    }
}

/**
 * Policy Request Instance Configuration
 */
export interface PolicyRequestInstanceConfig {
    subject: Subject;
    clientName: string;
    ipVerify?: boolean;
}

/**
 * Policy Request Class
 */
export class PolicyRequest {
    private subject: Subject;
    private clientName: string;
    private ipVerify: boolean;

    /**
     * Create PolicyRequest instance
     * @param config - Instance configuration with subject and clientName
     */
    constructor(config: PolicyRequestInstanceConfig) {
        this.subject = config.subject;
        this.clientName = config.clientName;
        this.ipVerify = config.ipVerify ?? false;
    }

    /**
     * Get current subject
     */
    getSubject(): Subject {
        return { ...this.subject };
    }

    /**
     * Get client name
     */
    getClientName(): string {
        return this.clientName;
    }

    /**
     * Get IP Verify
     */
    getIpVerify(): boolean {
        return this.ipVerify;
    }

    /**
   * Set ipVerify status
   * @param ipVerify - IP verification flag
   */
    setIpVerify(ipVerify: boolean): void {
        this.ipVerify = ipVerify;
    }

    /**
     * Update subject data
     * @param subject - New subject data (merges with existing)
     */
    updateSubject(subject: Subject): void {
        this.subject = { ...this.subject, ...subject };
    }

    /**
     * Create policy request and encode to Base64
     * @param config - Policy configuration object
     * @returns Base64 encoded string
     */
    async create(config: Omit<PolicyConfig, 'subject'> = {}): Promise<string> {
        let environment: Environment = {
            "resource": {
                "type": "WEB",
                "name": this.clientName,
                "timestamp": new Date(),
            }
        }

        if (this.ipVerify) {
            environment["public_ip"] = await getPublicIP();
        }

        const data: PolicyRequestData = {
            subject: this.subject,
            resource: {
                type: config.resource?.type || '',
                name: config.resource?.name || '',
                param: {
                    path: config.resource?.param?.path || '',
                    ...config.resource?.param,
                },
                ...config.resource,
            },
            action: config.action || '',
            environment: environment,
        };

        const jsonString = JSON.stringify(data);
        return base64Encode(jsonString);
    }

    /**
   * Create policy request headers
   * @param url - API url
   * @param method - HTTP method for call API
   * @param headerKey - Custom header key name (default: 'X-Policy-Request')
   * @returns Headers object
   */
    async createHeaders(
        url: string,
        method: HttpMethod,
        headerKey: string = 'X-Policy-Request'
    ): Promise<Record<string, string>> {
        const policyConfig = {
            resource: {
                "type": "API",
                "name": "",
                "param": {
                    "path": new URL(url).pathname
                }
            },
            action: method as string,
        }
        const encodedPolicy = await this.create(policyConfig);
        return {
            [headerKey]: encodedPolicy,
        };
    }

    /**
     * Decode Base64 policy request back to object
     * @param base64String - Base64 encoded policy request
     * @returns Decoded policy data
     */
    static decode(base64String: string): PolicyRequestData {
        try {
            const decoded = base64Decode(base64String);
            return JSON.parse(decoded);
        } catch (error) {
            throw new Error(
                `Invalid policy request format: ${error instanceof Error ? error.message : 'Unknown error'}`
            );
        }
    }

    /**
     * Validate policy request format
     * @param data - Policy data to validate
     * @returns true if valid, false otherwise
     */
    static validate(data: any): boolean {
        if (!data || typeof data !== 'object') {
            return false;
        }

        // Check required fields exist
        if (!('subject' in data) || !('resource' in data) || !('action' in data) || !('environment' in data)) {
            return false;
        }

        // Check subject and environment are objects
        if (typeof data.subject !== 'object' || typeof data.environment !== 'object') {
            return false;
        }

        // Check resource structure
        if (typeof data.resource !== 'object') {
            return false;
        }

        // Check action is string
        if (typeof data.action !== 'string') {
            return false;
        }

        return true;
    }

}

// Default export
export default PolicyRequest;