import axios from 'axios';

/**
 * Resource Parameter Structure
 */
export interface ResourceParam {
    path?: string;
    [key: string]: any;
}

/**
 * HTTP Method Enum
 */
export enum HttpMethod {
    GET = 'GET',
    POST = 'POST',
    PUT = 'PUT',
    DELETE = 'DELETE',
}

export enum ClientType {
    WEB = 'WEB',
    NATIVE = 'NATIVE',
}

/**
 * Resource Structure
 */
export interface Resource {
    type?: string;
    name?: string;
    param?: ResourceParam;
    [key: string]: any;
}

/**
 * Subject Structure
 */
export interface Subject {
    [key: string]: any;
}

/**
 * Environment Structure
 */
export interface Environment {
    [key: string]: any;
}

/**
 * Policy Request Configuration
 */
export interface PolicyConfig {
    resource?: Resource;
    action?: string;
}

/**
 * Policy Request Data Structure
 */
export interface PolicyRequestData {
    subject: Subject;
    resource: Resource;
    action: string;
    environment: Environment;
}

/**
 * Get public IP address
 * @returns Public IP address or null if failed
 */
async function getPublicIP(): Promise<string | null> {
    try {
        const response = await axios.get('https://api.ipify.org/?format=json', {
            timeout: 5000, // 5 second timeout
        });
        const ip = response.data?.ip;
        if (!ip || typeof ip !== 'string') {
            throw new Error('IP address not found in response');
        }
        return ip;
    } catch (error) {
        console.error('Failed to get public IP:', error);
        return null;
    }
}

// UTF-8 safe Base64 encode/decode helpers (Node + browser)
function base64EncodeUtf8(input: string): string {
    try {
        if (typeof Buffer !== 'undefined' && typeof Buffer.from === 'function') {
            return Buffer.from(input, 'utf8').toString('base64');
        }
        const btoaFn = (globalThis as any).btoa;
        if (typeof btoaFn === 'function') {
            // encodeURIComponent -> percent-encode -> unescape to binary string
            return btoaFn(unescape(encodeURIComponent(input)));
        }
    } catch (e) {
        // fallthrough to throw below
    }
    throw new Error('No available method to perform base64 encoding');
}

function base64DecodeUtf8(b64: string): string {
    try {
        if (typeof Buffer !== 'undefined' && typeof Buffer.from === 'function') {
            return Buffer.from(b64, 'base64').toString('utf8');
        }
        const atobFn = (globalThis as any).atob;
        if (typeof atobFn === 'function') {
            const binary = atobFn(b64);
            // convert binary string to percent-encoded string, then decode
            const percentEncoded = Array.prototype.map.call(binary, (ch: string) => {
                const code = ch.charCodeAt(0).toString(16).toUpperCase();
                return '%' + (code.length < 2 ? '0' + code : code);
            }).join('');
            return decodeURIComponent(percentEncoded);
        }
    } catch (e) {
        // fallthrough to throw below
    }
    throw new Error('No available method to perform base64 decoding');
}

/**
 * Policy Request Instance Configuration
 */
export interface PolicyRequestInstanceConfig {
    subject: Subject;
    clientName: string;
    clientType?: ClientType
    ipVerify?: boolean;
}

/**
 * Policy Request Class
 */
export class PolicyRequest {
    private subject: Subject;
    private clientName: string;
    private clientType: string;
    private ipVerify: boolean;

    /**
     * Create PolicyRequest instance
     * @param config - Instance configuration with subject and clientName
     */
    constructor(config: PolicyRequestInstanceConfig) {
        this.subject = config.subject;
        this.clientName = config.clientName;
        this.clientType = config.clientType ?? ClientType.WEB as string
        this.ipVerify = config.ipVerify ?? false;
    }

    /**
     * Get current subject
     */
    getSubject(): Subject {
        return { ...this.subject };
    }

    /**
     * Get client name
     */
    getClientName(): string {
        return this.clientName;
    }

    /**
     * Get client type
     */
    getClientType(): string {
        return this.clientType;
    }

     /**
   * Set client type
   * @param clientType - client type
   */
    setClirntType(clientType: string): void {
        this.clientType = clientType;
    }

    /**
     * Get IP Verify
     */
    getIpVerify(): boolean {
        return this.ipVerify;
    }

    /**
   * Set ipVerify status
   * @param ipVerify - IP verification flag
   */
    setIpVerify(ipVerify: boolean): void {
        this.ipVerify = ipVerify;
    }

    /**
     * Update subject data
     * @param subject - New subject data (merges with existing)
     */
    updateSubject(subject: Subject): void {
        this.subject = { ...this.subject, ...subject };
    }

    /**
     * Create policy request and encode to Base64
     * @param config - Policy configuration object
     * @returns Base64 encoded string
     */
    async create(config: Omit<PolicyConfig, 'subject'> = {}): Promise<string> {
        let environment: Environment = {
            "resource": {
                "type": this.clientType,
                "name": this.clientName,
                "timestamp": new Date(),
            }
        }

        if (this.ipVerify) {
            environment["public_ip"] = await getPublicIP();
        }

        const data: PolicyRequestData = {
            subject: this.subject,
            resource: {
                type: config.resource?.type || '',
                name: config.resource?.name || '',
                param: {
                    path: config.resource?.param?.path || '',
                    ...config.resource?.param,
                },
                ...config.resource,
            },
            action: config.action || '',
            environment: environment,
        };

        const jsonString = JSON.stringify(data);
        return base64EncodeUtf8(jsonString);
    }

    /**
   * Create policy request headers
   * @param url - API url
   * @param method - HTTP method for call API
   * @param headerKey - Custom header key name (default: 'X-Policy-Request')
   * @returns Headers object
   */
    async createHeaders(
        url: string,
        method: HttpMethod,
        headerKey: string = 'X-Policy-Request'
    ): Promise<Record<string, string>> {
        const policyConfig = {
            resource: {
                "type": "API",
                "name": "",
                "param": {
                    "path": new URL(url).pathname
                }
            },
            action: method as string,
        }
        const encodedPolicy = await this.create(policyConfig);
        return {
            [headerKey]: encodedPolicy,
        };
    }

    /**
     * Decode Base64 policy request back to object
     * @param base64String - Base64 encoded policy request
     * @returns Decoded policy data
     */
    static decode(base64String: string): PolicyRequestData {
        try {
            const decoded = base64DecodeUtf8(base64String);
            return JSON.parse(decoded);
        } catch (error) {
            throw new Error(
                `Invalid policy request format: ${error instanceof Error ? error.message : 'Unknown error'}`
            );
        }
    }

    /**
     * Validate policy request format
     * @param data - Policy data to validate
     * @returns true if valid, false otherwise
     */
    static validate(data: any): boolean {
        if (!data || typeof data !== 'object') {
            return false;
        }

        // Check required fields exist
        if (!('subject' in data) || !('resource' in data) || !('action' in data) || !('environment' in data)) {
            return false;
        }

        // Check subject and environment are objects
        if (typeof data.subject !== 'object' || typeof data.environment !== 'object') {
            return false;
        }

        // Check resource structure
        if (typeof data.resource !== 'object') {
            return false;
        }

        // Check action is string
        if (typeof data.action !== 'string') {
            return false;
        }

        return true;
    }

}

// Default export
export default PolicyRequest;