from typing import TypedDict,Union,Literal,Optional

from .strategies.gmail import Gmail,MessageParams as GmailMessageParams
from .strategies.telegram import Telegram, MessageParams as TelegramMessageParams
from .notification_sender import NotificationSender


class GmailConfig(TypedDict):
    smtp_server:str
    smtp_port:str
    sender_email : str
    sender_password:str
    
class TelegramConfig(TypedDict):
    bot_token:str
    chat_id : str

class NotificationConfig(TypedDict):
    gmail : Optional[GmailConfig]
    telegram : Optional[TelegramConfig]
    
    
class NotificationMessageParams(TypedDict):
    method: Literal['gmail', 'telegram']
    params: Union[GmailMessageParams, TelegramMessageParams]
    
class NotificationCenter:
    _config : NotificationConfig
    _gmail_strategy : Optional[Gmail] = None
    _telegram_strategy : Optional[Telegram] = None
    _sender  : NotificationSender
    
    def __init__(self,config:NotificationConfig):
        self._config = config
        self._sender = NotificationSender()
        
        if config.get("gmail") :
            self._gmail_strategy = Gmail(
                smtp_server=config["gmail"]["smtp_server"],
                smtp_port=config["gmail"]["smtp_port"]
            )
            self._gmail_strategy.set_sender(
                mail=config["gmail"]['sender_email'],
                password=config["gmail"]['sender_password']
            )
        if config.get("telegram"):
            self._telegram_strategy = Telegram(
                bot_token=config["telegram"]["bot_token"],
                chat_id=config["telegram"]["chat_id"],
            )
            
    def send_notification(self,params:NotificationMessageParams,raise_error:Optional[bool]=False):
        method = params.get("method")
        if method == "gmail" and self._gmail_strategy :
            self._sender.set_strategy(self._gmail_strategy)
        
        elif method == "telegram" and self._telegram_strategy :
            self._sender.set_strategy(self._telegram_strategy)
        
        else :
            raise ValueError(f"Unsupported method: {method}")
        
        self._sender.send_notification(params["params"],raise_error)