import React, { useCallback, useState } from 'react';
import ReactFlow, {
    MiniMap,
    Controls,
    Background,
    useNodesState,
    useEdgesState,
    addEdge,
    Panel,
    useReactFlow,
} from 'reactflow';

import 'reactflow/dist/style.css';

const flowKey = 'example-flow';

const getNodeId = () => `randomnode_${+new Date()}`;

const initialNodes = [
    { id: '1', data: { label: 'Node 1' }, position: { x: 100, y: 100 } },
    { id: '2', data: { label: 'Node 2' }, position: { x: 100, y: 200 } },
];

const initialEdges = [{ id: 'e1-2', source: '1', target: '2' }];

const DesignManagement = () => {

    const [nodes, setNodes, onNodesChange] = useNodesState(initialNodes);
    const [edges, setEdges, onEdgesChange] = useEdgesState(initialEdges);
    const [rfInstance, setRfInstance] = useState(null);
    const { setViewport } = useReactFlow();
    const [selectedFile, setSelectedFile] = useState(null);
    const [selectedJsonFile, setSelectedJsonFile] = useState(null);

    const onConnect = useCallback((params) => setEdges((eds) => addEdge(params, eds)), [setEdges]);
    const onSave = useCallback(() => {
        if (rfInstance) {
            const flow = rfInstance.toObject();
            // console.log('flowKey: ' + flowKey)
            // console.log('flow: ' + flow)
            localStorage.setItem(flowKey, JSON.stringify(flow));
        }
    }, [rfInstance]);

    const onRestore = useCallback(() => {
        const restoreFlow = async () => {
            const flow = JSON.parse(localStorage.getItem(flowKey));

            if (flow) {
                const { x = 0, y = 0, zoom = 1 } = flow.viewport;
                setNodes(flow.nodes || []);
                setEdges(flow.edges || []);
                setViewport({ x, y, zoom });
            }
        };

        restoreFlow();
    }, [setNodes, setViewport]);

    const onAdd = useCallback(() => {
        const newNode = {
            id: getNodeId(),
            data: { label: 'Added node' },
            position: {
                x: Math.random() * window.innerWidth - 100,
                y: Math.random() * window.innerHeight,
            },
        };
        setNodes((nds) => nds.concat(newNode));
    }, [setNodes]);

    const onExport = useCallback(() => {
        if (rfInstance) {
            const flow = rfInstance.toObject();
            // console.log("flow:", JSON.stringify(flow, null, 2));

            // สร้าง Blob จาก JSON string
            const jsonString = JSON.stringify(flow, null, 2);

            const blob = new Blob([jsonString], { type: 'application/json' });

            // สร้าง URL จาก Blob
            const url = URL.createObjectURL(blob);

            // สร้างลิงก์ดาวน์โหลดชั่วคราว
            const link = document.createElement('a');
            link.href = url;
            link.download = 'data-react-flow.json';

            // เพิ่มลิงก์ไปยัง DOM และคลิกมันโปรแกรมมิ่ง
            document.body.appendChild(link);
            link.click();

            // ลบลิงก์ออกจาก DOM
            document.body.removeChild(link);

            // ปลดปล่อยหน่วยความจำ URL ที่สร้างขึ้น
            URL.revokeObjectURL(url);
        }
    }, [rfInstance]);

    const handleFileChange = (event) => {
        const file = event.target.files[0];
        const reader = new FileReader();

        reader.onload = (event) => {
            try {
                const json = JSON.parse(event.target.result);
                console.log("json file:", json);
                setSelectedJsonFile(json)
            } catch (error) {
                console.error("Error parsing JSON: ", error);
            }
        };

        if (file) {
            reader.readAsText(file);
        }
        setSelectedFile(file);
    };

    const handleUploadClick = () => {
        document.getElementById('fileInput').click();
    };

    const handleSubmit = () => {
        if (!selectedFile) {
            alert('Please select a file first!');
            return;
        }

        // console.log("selectedFile:", selectedFile);
        // console.log("file:", JSON.stringify(selectedFile, null, 2));
        // สามารถเพิ่มการประมวลผลไฟล์ที่เลือกได้ที่นี่
        // เช่น อัปโหลดไฟล์ไปยังเซิร์ฟเวอร์

        const renderFlow = async () => {
            console.log("selectedJsonFile:", selectedJsonFile);

            if (selectedJsonFile) {
                const { x = 0, y = 0, zoom = 1 } = selectedJsonFile.viewport;
                setNodes(selectedJsonFile.nodes || []);
                setEdges(selectedJsonFile.edges || []);
                setViewport({ x, y, zoom });
            }
        };

        renderFlow();
    };

    const handleCancel = () => {
        setSelectedFile(null)
    };

    return (
        <div style={{ width: '100%', height: '100%' }}>
            <ReactFlow
                nodes={nodes}
                edges={edges}
                onNodesChange={onNodesChange}
                onEdgesChange={onEdgesChange}
                onConnect={onConnect}
                onInit={setRfInstance}
            >
                <Panel position="top-left">
                    <div style={{ display: 'flex', flexDirection: 'row', maxHeight: 20 }}>
                        <button onClick={handleUploadClick}>import</button>
                        <input
                            type="file"
                            id="fileInput"
                            style={{ display: 'none' }}
                            onChange={handleFileChange}
                        />
                        {selectedFile && <div style={{ marginLeft: 10, marginRight: 10 }}>file: {selectedFile?.name}</div>}
                        <button onClick={handleSubmit}>upload</button>
                        <button onClick={handleCancel}>cancel</button>
                    </div>
                </Panel>
                <Panel position="top-right">
                    <button onClick={onSave}>save</button>
                    <button onClick={onRestore}>restore</button>
                    <button onClick={onAdd}>add node</button>
                    <button onClick={onExport}>export</button>
                </Panel>
                <Controls />
                <MiniMap />
                <Background variant="dots" gap={12} size={1} />
            </ReactFlow>
        </div>
    );
}

export default DesignManagement;